/**
 * CarbonGuru Browser Extension - Popup Script
 * Handles the popup UI interactions and communication with background script
 * Updated to automatically load current page URL and use web app backend functions
 */

/**
 * Format cost with smart decimal precision
 * 4 decimals for sub-cent amounts (< $0.01), 2 decimals otherwise
 * @param {number|string} cost - The cost value to format
 * @returns {string} Formatted cost string with $ prefix
 */
const formatCost = (cost) => {
    const amount = parseFloat(cost) || 0;
    if (amount <= 0) return '$0.00';
    if (amount < 0.01) return '$' + amount.toFixed(4);
    return '$' + amount.toFixed(2);
};

void 0;

class CarbonGuruPopup {
    constructor() {
        void 0;
        this.currentTab = null;
        this.currentProductData = null;
        this.currentUrl = null;
        this.environmentConfig = null;
        try {
            this.init();
        } catch (error) {
            console.error('CarbonGuru: Error in constructor init:', error);
        }
    }

    async loadEnvironmentConfig() {
        return new Promise((resolve) => {
            browser.runtime.sendMessage({action: 'getEnvironmentConfig'}, (response) => {
                this.environmentConfig = response;
                void 0;
                resolve();
            });
        });
    }

    async getBackendUrl() {
        if (!this.environmentConfig) {
            await this.loadEnvironmentConfig();
        }
        return this.environmentConfig.backendUrl;
    }

    async getProductUrl(path) {
        return new Promise((resolve) => {
            browser.runtime.sendMessage({
                action: 'getProductUrl', 
                path: path
            }, (response) => {
                resolve(response.url);
            });
        });
    }

    async init() {
        try {
            void 0;

            // Load environment configuration first
            await this.loadEnvironmentConfig();

            void 0;
            void 0;

            // Ensure DOM is ready before setting up event listeners and loading UI elements
            if (document.readyState === 'loading') {
                void 0;
                document.addEventListener('DOMContentLoaded', async () => {
                    void 0;
                    // Hide analysis results section on popup load
                    const resultsSection = document.getElementById('results');
                    if (resultsSection) resultsSection.style.display = 'none';

                    this.setupEventListeners();
                    await this.loadBasket();
                    await this.loadRecentItems();
                    this.loadImpactSection(); // Non-blocking, don't await
                    // Automatically sync with WooCommerce on popup open
                    await this.syncOnPopupOpen();
                });
            } else {
                void 0;
                // Hide analysis results section on popup load
                const resultsSection = document.getElementById('results');
                if (resultsSection) resultsSection.style.display = 'none';

                this.setupEventListeners();
                await this.loadBasket();
                await this.loadRecentItems();
                this.loadImpactSection(); // Non-blocking, don't await
                // Automatically sync with WooCommerce on popup open
                await this.syncOnPopupOpen();
            }

            this.setupMessageListener();
            await this.updatePageInfo();
            
            // Load version display immediately
            this.loadVersionDisplay();

            // Also set a simple footer if present
            try {
                const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
                const resp = await runtime.sendMessage({ action: 'getVersionInfo' });
                const footer = document.querySelector('#version-footer');
                if (footer && resp && resp.currentVersion) {
                    // Include build number if present (Safari uses separate build_number field)
                    let versionStr = resp.currentVersion;
                    if (resp.buildNumber && !versionStr.includes(resp.buildNumber)) {
                        versionStr = `${versionStr}.${resp.buildNumber}`;
                    }
                    footer.textContent = `v${versionStr} (${resp.isDevelopment ? 'Dev' : 'Prod'})`;
                }
            } catch (_) {}
            
            void 0;
            
            // Force admin functions to load after a short delay
            setTimeout(() => {
                void 0;
                this.loadVersionInfo();
                this.checkAuthStatus();
                this.updateSessionStatus();
            }, 1000);

            // Start periodic session status updates (every 3 seconds)
            this.sessionStatusInterval = setInterval(() => {
                this.updateSessionStatus();
            }, 3000);
        } catch (error) {
            console.error('CarbonGuru: Error in init():', error);
        }
    }

    setupEventListeners() {
        try {
            void 0;
            void 0;
            

        
        // Manual URL input
        const analyzeManualUrlBtn = document.getElementById('analyze-manual-url-btn');
        if (analyzeManualUrlBtn) {
            analyzeManualUrlBtn.addEventListener('click', () => {
                void 0;
                this.analyzeManualUrl();
            });
        }
        
        const manualUrlInput = document.getElementById('manual-url-input');
        if (manualUrlInput) {
            manualUrlInput.addEventListener('keypress', (e) => {
                if (e.key === 'Enter') {
                    void 0;
                    this.analyzeManualUrl();
                }
            });
        }

        // Product detection actions
        const selectDifferentBtn = document.getElementById('select-different');
        if (selectDifferentBtn) {
            selectDifferentBtn.addEventListener('click', () => {
                void 0;
                this.selectDifferentProduct();
            });
        }
        
        const manualUrlToggleBtn = document.getElementById('manual-url-toggle');
        if (manualUrlToggleBtn) {
            manualUrlToggleBtn.addEventListener('click', () => {
                void 0;
                this.toggleManualUrlInput();
            });
        }
        
        // Minimize/Restore/Close functionality
        const minimizeBtn = document.getElementById('minimize-btn');
        void 0;
        if (minimizeBtn) {
            void 0;
            minimizeBtn.addEventListener('click', () => {
                void 0;
                this.minimizePopup();
            });
        } else {
            console.error('CarbonGuru: minimize-btn element not found!');
        }
        
        const restoreBtn = document.getElementById('restore-btn');
        if (restoreBtn) {
            restoreBtn.addEventListener('click', () => {
                void 0;
                this.restorePopup();
            });
        }
        
        const closeBtn = document.getElementById('close-btn');
        void 0;
        if (closeBtn) {
            void 0;
            closeBtn.addEventListener('click', () => {
                void 0;
                this.closePopup();
            });
        } else {
            console.error('CarbonGuru: close-btn element not found!');
        }
        
        const minimizedAnalyzeBtn = document.getElementById('minimized-analyze-btn');
        if (minimizedAnalyzeBtn) {
            minimizedAnalyzeBtn.addEventListener('click', () => {
                void 0;
                this.extractWithFullPage();
            });
        }

        // Main "Analyze Carbon Footprint" button in product detection section
        const startAnalysisBtn = document.getElementById('start-analysis');
        if (startAnalysisBtn) {
            startAnalysisBtn.addEventListener('click', () => {
                void 0;
                this.extractWithFullPage();
            });
        }

        // Admin section toggle
        const adminToggle = document.getElementById('admin-toggle');
        if (adminToggle) {
            adminToggle.addEventListener('click', () => {
                void 0;
                this.toggleAdminSection();
            });
            adminToggle.addEventListener('keydown', (e) => {
                if (e.key === 'Enter' || e.key === ' ') {
                    e.preventDefault();
                    this.toggleAdminSection();
                }
            });
        }

        // Admin functions
        const authBtn = document.getElementById('auth-btn');
        if (authBtn) {
            authBtn.addEventListener('click', () => {
                void 0;
                this.initiateOAuth();
            });
        }

        // Logout button handler
        const logoutBtn = document.getElementById('logout-btn');
        if (logoutBtn) {
            logoutBtn.addEventListener('click', () => {
                void 0;
                this.logout();
            });
        }

        const updateBtn = document.getElementById('update-btn');
        if (updateBtn) {
            updateBtn.addEventListener('click', () => {
                void 0;
                this.checkForUpdates();
            });
        }

        const dataConsentLink = document.getElementById('data-consent-link');
        if (dataConsentLink) {
            dataConsentLink.addEventListener('click', (e) => {
                e.preventDefault();
                const runtimeApi = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
                const tabsApi = typeof browser !== 'undefined' ? browser.tabs : chrome.tabs;
                tabsApi.create({ url: runtimeApi.getURL('consent.html') });
            });
        }

        const cancelAnalysisBtn = document.getElementById('cancel-analysis-btn');
        if (cancelAnalysisBtn) {
            cancelAnalysisBtn.addEventListener('click', () => {
                void 0;
                this.cancelAnalysis();
            });
        }

        // Offset Basket buttons
        const offsetAllBtn = document.getElementById('offset-all-items-btn');
        if (offsetAllBtn) {
            offsetAllBtn.addEventListener('click', () => {
                void 0;
                this.offsetAllItems();
            });
        }

        const clearBasketBtn = document.getElementById('clear-basket-btn');
        if (clearBasketBtn) {
            clearBasketBtn.addEventListener('click', () => {
                void 0;
                this.clearBasket();
            });
        }

        const syncToWooCommerceBtn = document.getElementById('sync-to-woocommerce-btn');
        if (syncToWooCommerceBtn) {
            syncToWooCommerceBtn.addEventListener('click', () => {
                void 0;
                this.syncToWooCommerce();
            });
        }

        // View Offset Cart button
        const viewOffsetCartBtn = document.getElementById('view-offset-cart-btn');
        if (viewOffsetCartBtn) {
            viewOffsetCartBtn.addEventListener('click', () => {
                void 0;
                this.openTypeSpecificCart('offset');
            });
        }

        // Checkout Offset button
        const checkoutOffsetBtn = document.getElementById('checkout-offset-btn');
        if (checkoutOffsetBtn) {
            checkoutOffsetBtn.addEventListener('click', () => {
                void 0;
                this.openTypeSpecificCheckout('offset');
            });
        }

        // View Removal Cart button
        const viewRemovalCartBtn = document.getElementById('view-removal-cart-btn');
        if (viewRemovalCartBtn) {
            viewRemovalCartBtn.addEventListener('click', () => {
                void 0;
                this.openTypeSpecificCart('removal');
            });
        }

        // Checkout Removal button
        const checkoutRemovalBtn = document.getElementById('checkout-removal-btn');
        if (checkoutRemovalBtn) {
            checkoutRemovalBtn.addEventListener('click', () => {
                void 0;
                this.openTypeSpecificCheckout('removal');
            });
        }

        // Manual sync button
        const manualSyncBtn = document.getElementById('manual-sync-btn');
        if (manualSyncBtn) {
            manualSyncBtn.addEventListener('click', () => {
                void 0;
                this.performManualSync();
            });
        }

        void 0;
        } catch (error) {
            console.error('CarbonGuru: Error setting up event listeners:', error);
        }
    }

    setupMessageListener() {
        // Listen for analysis completion messages from background script
        const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
        runtime.onMessage.addListener((message, sender, sendResponse) => {
            void 0;

            if (message.action === 'analysisComplete') {
                void 0;
                this.updateAnalysisResults(message);
            }

            if (message.action === 'authStatusChanged') {
                void 0;
                this.checkAuthStatus();
            }

            return false; // Don't keep message channel open
        });
    }

    updateAnalysisResults(message) {
        const resultsSection = document.getElementById('results');
        const resultsContent = document.getElementById('results-content');
        
        if (message.success && message.data) {
            // Show results section
            resultsSection.style.display = 'block';
            
            const data = message.data;
            const footprint = data.product_carbon_footprint && data.product_carbon_footprint > 0 ?
                `${data.product_carbon_footprint.toFixed(2)} kg CO2e` : 'Not calculated';
            
            // Determine classification based on available data
            let classification = '';
            if (data.sector_code && data.sector_name) {
                // Open CEDA / Global processing
                classification = `Sector: ${data.sector_code} - ${data.sector_name} (${data.country_code || 'Global'})`;
            } else if (data.naics_code && data.naics_title) {
                // US NAICS / EIO processing  
                classification = `NAICS: ${data.naics_code} - ${data.naics_title}`;
            } else if (data.naics_code) {
                // NAICS code only
                classification = `NAICS: ${data.naics_code}`;
            } else {
                classification = 'No classification mapping';
            }
            
            // Display offset and removal costs on action buttons (with sub-cent precision)
            const offsetCostDisplay = data.offset_cost_usd ? formatCost(data.offset_cost_usd) : '';
            const removalCostDisplay = data.removal_cost_usd ? formatCost(data.removal_cost_usd) : '';

            const resultHTML = `
                <div class="analysis-result">
                    <div class="result-header">
                        <strong>${data.product_name || 'Current Product'}</strong>
                    </div>
                    <div class="result-footprint">
                        <span class="label">Carbon Footprint:</span> ${footprint}
                    </div>
                    <div class="result-classification">
                        <span class="label">Industry Classification:</span> ${classification}
                    </div>
                    <div class="result-actions">
                        <div class="btn-group" style="display: flex; gap: 8px; margin-bottom: 8px;">
                            <button class="btn-small offset-btn add-to-basket-btn" data-item='${JSON.stringify(data)}' data-preference-type="offset">
                                Offset Now ${offsetCostDisplay ? `<span class="btn-cost">${offsetCostDisplay}</span>` : ''}
                            </button>
                            <button class="btn-small removal-btn add-to-basket-btn" data-item='${JSON.stringify(data)}' data-preference-type="removal">
                                Remove Now ${removalCostDisplay ? `<span class="btn-cost">${removalCostDisplay}</span>` : ''}
                            </button>
                        </div>
                        <a href="${data.details_url}" target="_blank" class="analysis-btn details-link">View details</a>
                    </div>
                </div>
            `;
            
            // Update results section
            resultsContent.innerHTML = DOMPurify.sanitize(resultHTML);
            
            // Store current product data for minimized view
            this.currentProductData = data;
            
        } else {
            const errorHTML = `
                <div class="error-message">
                    Analysis failed: ${message.error || 'Unknown error'}
                </div>
            `;
            
            resultsSection.style.display = 'block';
            resultsContent.innerHTML = DOMPurify.sanitize(errorHTML);
        }
        
        // Also refresh recent items to show the new result
        this.loadRecentItems();
    }

    minimizePopup() {
        const mainContent = document.getElementById('main-content');
        const minimizedView = document.getElementById('minimized-view');
        const header = document.querySelector('.header');
        const pageInfo = document.getElementById('page-info');

        if (mainContent && minimizedView) {
            mainContent.style.display = 'none';
            if (header) header.style.display = 'none';
            if (pageInfo) pageInfo.style.display = 'none';
            // Impact section stays visible in minimized view
            minimizedView.style.display = 'block';
        }
    }

    restorePopup() {
        const mainContent = document.getElementById('main-content');
        const minimizedView = document.getElementById('minimized-view');
        const header = document.querySelector('.header');

        if (mainContent && minimizedView) {
            mainContent.style.display = 'block';
            if (header) header.style.display = '';
            minimizedView.style.display = 'none';
        }
    }

    closePopup() {
        // Close the popup window
        window.close();
    }

    async updatePageInfo() {
        try {
            void 0;
            
            // Use browser APIs for Firefox compatibility
            const tabsAPI = typeof browser !== 'undefined' ? browser.tabs : chrome.tabs;
            
            // Debug: Check if tabs API is available
            if (!tabsAPI) {
                console.error('CarbonGuru: tabs API is not available');
                this.showError('Extension permissions not available');
                return;
            }
            
            void 0;
            
            // Try multiple approaches to get the current tab
            let tabs = null;
            let currentTab = null;
            
            // Approach 1: Standard query
            try {
                tabs = await tabsAPI.query({ active: true, currentWindow: true });
                void 0;
            } catch (error) {
                console.error('CarbonGuru: Standard tab query failed:', error);
            }
            
            // Approach 2: If standard query failed, try with more specific parameters
            if (!tabs || tabs.length === 0) {
                try {
                    void 0;
                    tabs = await tabsAPI.query({ 
                        active: true, 
                        currentWindow: true,
                        status: 'complete'
                    });
                    void 0;
                } catch (error) {
                    console.error('CarbonGuru: Alternative tab query failed:', error);
                }
            }
            
            // Approach 3: If still no tabs, try getting the active tab directly
            if (!tabs || tabs.length === 0) {
                try {
                    void 0;
                    const activeTab = await tabsAPI.getCurrent();
                    if (activeTab) {
                        tabs = [activeTab];
                        void 0;
                    }
                } catch (error) {
                    console.error('CarbonGuru: Direct tab access failed:', error);
                }
            }
            
            // Approach 4: Last resort - try to get any tab
            if (!tabs || tabs.length === 0) {
                try {
                    void 0;
                    tabs = await tabsAPI.query({});
                    void 0;
                    // Find the active tab from all tabs
                    if (tabs && tabs.length > 0) {
                        const activeTab = tabs.find(tab => tab.active);
                        if (activeTab) {
                            tabs = [activeTab];
                            void 0;
                        }
                    }
                } catch (error) {
                    console.error('CarbonGuru: All tabs query failed:', error);
                }
            }
            
            if (!tabs || tabs.length === 0) {
                console.error('CarbonGuru: All tab query approaches failed');
                
                // Fallback: Try to get URL from content script
                void 0;
                try {
                    // Try to send message to content script to get current URL
                    const response = await tabsAPI.sendMessage(this.currentTab?.id || 'unknown', {
                        action: 'getCurrentUrl'
                    });
                    
                    if (response && response.url) {
                        void 0;
                        this.currentUrl = response.url;
                        
                        // Update display
                        const currentUrlElement = document.getElementById('current-url');
                        if (currentUrlElement) {
                            currentUrlElement.textContent = this.currentUrl;
                        }
                        
                        // Update validation message
                        const validationMessage = document.getElementById('url-validation-message');
                        if (validationMessage) {
                            if (this.isProductPage(this.currentUrl)) {
                                validationMessage.textContent = 'This appears to be a product page. Click "Validate & Extract" to analyze.';
                                validationMessage.className = 'validation-message success';
                            } else {
                                validationMessage.textContent = 'This may not be a product page. You can still try to extract data or use manual URL entry.';
                                validationMessage.className = 'validation-message warning';
                            }
                        }
                        return;
                    }
                } catch (error) {
                    console.error('CarbonGuru: Content script fallback failed:', error);
                }
                
                this.showError('No active tab found - please refresh the page and try again');
                return;
            }
            
            currentTab = tabs[0];
            this.currentTab = currentTab;
            this.currentUrl = currentTab.url;
            void 0;
            void 0;

            // Update current URL display
            const currentUrlElement = document.getElementById('current-url');
            if (currentUrlElement) {
                currentUrlElement.textContent = this.currentUrl;
                void 0;
            } else {
                console.error('CarbonGuru: Current URL element not found');
            }

            // Update validation message based on URL type
            const validationMessage = document.getElementById('url-validation-message');
            if (validationMessage) {
                if (this.isProductPage(this.currentUrl)) {
                    validationMessage.textContent = 'This appears to be a product page. Click "Validate & Extract" to analyze.';
                    validationMessage.className = 'validation-message success';
                } else {
                    validationMessage.textContent = 'This may not be a product page. You can still try to extract data or use manual URL entry.';
                    validationMessage.className = 'validation-message warning';
                }
            }

            // Page-type badge removed from popup - shown only by content script on page
            // await this.updatePageTypeBadge(this.currentUrl, currentTab.title);

            // Check if current page is a product page and auto-extract if possible
            if (this.isProductPage(this.currentUrl)) {
                void 0;
                // Don't auto-extract, let user choose when to validate
            } else {
                void 0;
            }
        } catch (error) {
            console.error('CarbonGuru: Error updating page info:', error);
            this.showError('Error loading page info: ' + error.message);
        }
    }

    async updatePageTypeBadge(url, pageTitle = '') {
        try {
            void 0;

            // Get references to UI elements
            const badgeElement = document.getElementById('page-type-badge');
            const pageInfoSection = document.getElementById('page-info');
            const documentTypeElement = document.getElementById('document-type');
            const orderNumberElement = document.getElementById('order-number');

            if (!badgeElement) {
                void 0;
                return;
            }

            // Send message to background script to check page type
            const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
            const response = await runtime.sendMessage({
                action: 'detectPageType',
                url: url,
                pageTitle: pageTitle
            });

            if (!response) {
                badgeElement.textContent = '❓ Unknown';
                badgeElement.className = 'badge badge-unknown';
                if (pageInfoSection) pageInfoSection.style.display = 'none';
                return;
            }

            // Handle order pages
            if (response.isOrderPage) {
                badgeElement.textContent = '📋 Order';
                badgeElement.className = 'badge badge-order';

                // Hide page info section (document type and order number moved to content script)
                if (pageInfoSection) pageInfoSection.style.display = 'none';

                // Keep elements hidden
                if (documentTypeElement) documentTypeElement.style.display = 'none';
                if (orderNumberElement) orderNumberElement.style.display = 'none';

                void 0;
                return;
            }

            // Handle shopping cart pages
            if (response.isCartPage) {
                badgeElement.textContent = '🛒 Cart';
                badgeElement.className = 'badge badge-cart';
                if (pageInfoSection) pageInfoSection.style.display = 'none';
                void 0;
                return;
            }

            // Handle product pages
            if (response.isProductPage) {
                badgeElement.textContent = '📦 Product';
                badgeElement.className = 'badge badge-product';
                if (pageInfoSection) pageInfoSection.style.display = 'none';
                void 0;
                return;
            }

            // Default: Unknown page type
            badgeElement.textContent = '❓ Unknown';
            badgeElement.className = 'badge badge-unknown';
            if (pageInfoSection) pageInfoSection.style.display = 'none';
            void 0;

        } catch (error) {
            console.error('CarbonGuru: Error updating page type badge:', error);
        }
    }

    showError(message) {
        console.error('CarbonGuru: Showing error:', message);
        const validationMessage = document.getElementById('url-validation-message');
        if (validationMessage) {
            validationMessage.textContent = message;
            validationMessage.className = 'validation-message error';
        }
        
        const currentUrlElement = document.getElementById('current-url');
        if (currentUrlElement) {
            currentUrlElement.textContent = 'Error loading URL';
        }
    }

    isProductPage(url) {
        const urlLower = url.toLowerCase();
        const productPatterns = [
            /\/dp\//,           // Amazon
            /\/product\//,      // General
            /\/item\//,         // Etsy, eBay
            /\/p\//,            // Target
            /\/products\//,     // Shopify
            /\/buy\//,          // Walmart
            /\/shop\//,         // General shop
            /\/store\//,        // General store
        ];
        
        return productPatterns.some(pattern => pattern.test(urlLower));
    }

    async validateAndExtractCurrentUrl() {
        void 0;
        void 0;
        void 0;
        
        if (!this.currentUrl) {
            void 0;
            void 0;
            
            // Use browser APIs for Firefox compatibility
            const tabsAPI = typeof browser !== 'undefined' ? browser.tabs : chrome.tabs;
            
            // Try to get the current URL again using the same robust approach
            try {
                let tabs = null;
                
                // Try multiple approaches
                try {
                    tabs = await tabsAPI.query({ active: true, currentWindow: true });
                } catch (error) {
                    console.error('CarbonGuru: Standard query failed:', error);
                }
                
                if (!tabs || tabs.length === 0) {
                    try {
                        tabs = await tabsAPI.query({ active: true, currentWindow: true, status: 'complete' });
                    } catch (error) {
                        console.error('CarbonGuru: Alternative query failed:', error);
                    }
                }
                
                if (!tabs || tabs.length === 0) {
                    try {
                        const allTabs = await tabsAPI.query({});
                        const activeTab = allTabs.find(tab => tab.active);
                        if (activeTab) {
                            tabs = [activeTab];
                        }
                    } catch (error) {
                        console.error('CarbonGuru: All tabs query failed:', error);
                    }
                }
                
                if (tabs && tabs.length > 0) {
                    this.currentTab = tabs[0];
                    this.currentUrl = this.currentTab.url;
                    void 0;
                } else {
                    alert('No current URL available - please refresh the page and try again');
                    return;
                }
            } catch (error) {
                console.error('CarbonGuru: Error refreshing current URL:', error);
                alert('Error getting current URL: ' + error.message);
                return;
            }
        }

        void 0;

        try {
            // Show loading state
            const validateBtn = document.getElementById('validate-url-btn');
            const originalText = validateBtn.innerHTML;
            validateBtn.disabled = true;
            validateBtn.innerHTML = DOMPurify.sanitize('<i class="fas fa-spinner fa-spin"></i> Validating...');

            // Step 1: Validate URL using web app's validate_url endpoint or cached result
            void 0;
            let validationResponse = await this.getCachedValidationResponse(this.currentUrl);
            
            if (!validationResponse) {
                void 0;
                validationResponse = await this.validateUrl(this.currentUrl);
            } else {
                void 0;
            }
            
            if (!validationResponse.success) {
                throw new Error(validationResponse.error || 'URL validation failed');
            }

            void 0;

            // Check for existing mapped items
            if (validationResponse.has_existing_items && validationResponse.existing_items && validationResponse.existing_items.length > 0) {
                void 0;
                
                // Reset button state first
                validateBtn.disabled = false;
                validateBtn.innerHTML = DOMPurify.sanitize(originalText);
                
                // Found existing mappings, continuing with analysis - user choice handled in content script
                void 0;
                
                // Continue with processing - user choice handled in content script
                validateBtn.disabled = true;
                validateBtn.innerHTML = DOMPurify.sanitize('<i class="fas fa-spinner fa-spin"></i> Processing...');
            }

            // Step 2: Process URL using web app's process_url endpoint
            void 0;
            validateBtn.innerHTML = DOMPurify.sanitize('<i class="fas fa-spinner fa-spin"></i> Processing...');
            
            const processResponse = await this.processUrl(this.currentUrl);
            
            if (!processResponse.success) {
                throw new Error(processResponse.error || 'URL processing failed');
            }

            void 0;
            
            // Step 3: Display the extracted product data
            this.currentProductData = processResponse.preview;
            this.showProductDetection(processResponse.preview);
            
            // Reset button
            validateBtn.disabled = false;
            validateBtn.innerHTML = DOMPurify.sanitize(originalText);

        } catch (error) {
            console.error('CarbonGuru: Error in validateAndExtractCurrentUrl:', error);
            
            // Reset button
            const validateBtn = document.getElementById('validate-url-btn');
            validateBtn.disabled = false;
            validateBtn.innerHTML = DOMPurify.sanitize('Validate & Extract');
            
            alert('Error processing URL: ' + error.message);
        }
    }

    async getCachedValidationResponse(url) {
        // Check background script cache first
        return new Promise((resolve) => {
            chrome.runtime.sendMessage({
                action: 'getCachedValidation',
                url: url
            }, (response) => {
                resolve(response || null);
            });
        });
    }

    async validateUrl(url) {
        void 0;
        
        try {
            const backendUrl = await this.getBackendUrl();
            const response = await fetch(`${backendUrl}/validate_url`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({ url: url })
            });

            if (!response.ok) {
                throw new Error(`HTTP error ${response.status}`);
            }

            const data = await response.json();
            void 0;
            
            return {
                success: !data.error,
                data: data,
                error: data.error,
                // Include deduplication info
                has_existing_items: data.has_existing_items,
                existing_items: data.existing_items,
                has_semantic_similarity: data.has_semantic_similarity,
                semantic_similarity: data.semantic_similarity,
                // Include domain info for status messages
                domain_info: data.domain_info
            };
        } catch (error) {
            console.error('CarbonGuru: Error validating URL:', error);
            return {
                success: false,
                error: error.message
            };
        }
    }

    async processUrl(url) {
        void 0;
        
        try {
            const backendUrl = await this.getBackendUrl();
            const response = await fetch(`${backendUrl}/process_url`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({ url: url })
            });

            if (!response.ok) {
                throw new Error(`HTTP error ${response.status}`);
            }

            const data = await response.json();
            void 0;
            
            return {
                success: !data.error,
                preview: data.preview,
                filepath: data.filepath,
                error: data.error
            };
        } catch (error) {
            console.error('CarbonGuru: Error processing URL:', error);
            return {
                success: false,
                error: error.message
            };
        }
    }

    async analyzeManualUrl() {
        void 0;
        const urlInput = document.getElementById('manual-url-input');
        const url = urlInput.value.trim();

        if (!url) {
            void 0;
            alert('Please enter a valid URL');
            return;
        }

        void 0;

        try {
            // Use the same validation and processing flow as current page (with caching)
            let validationResponse = await this.getCachedValidationResponse(url);
            
            if (!validationResponse) {
                void 0;
                validationResponse = await this.validateUrl(url);
            } else {
                void 0;
            }
            
            if (!validationResponse.success) {
                throw new Error(validationResponse.error || 'URL validation failed');
            }

            // Check for existing mapped items
            if (validationResponse.has_existing_items && validationResponse.existing_items && validationResponse.existing_items.length > 0) {
                void 0;
                
                // Found existing mappings, continuing with analysis - user choice handled in content script
                void 0;
                
                // Continue with processing - user choice handled in content script
                void 0;
            }

            const processResponse = await this.processUrl(url);
            
            if (!processResponse.success) {
                throw new Error(processResponse.error || 'URL processing failed');
            }

            this.currentProductData = processResponse.preview;
            this.showProductDetection(processResponse.preview);
            
        } catch (error) {
            console.error('CarbonGuru: Error analyzing manual URL:', error);
            alert('Error analyzing URL: ' + error.message);
        }
    }

    async extractWithFullPage() {
        void 0;
        void 0;
        void 0;
        

        
        if (!this.currentTab) {
            void 0;
            void 0;
            
            // Try to get the current tab again using the same robust approach
            try {
                let tabs = null;
                
                // Try multiple approaches
                try {
                    tabs = await tabsAPI.query({ active: true, currentWindow: true });
                } catch (error) {
                    console.error('CarbonGuru: Standard query failed:', error);
                }
                
                if (!tabs || tabs.length === 0) {
                    try {
                        tabs = await tabsAPI.query({ active: true, currentWindow: true, status: 'complete' });
                    } catch (error) {
                        console.error('CarbonGuru: Alternative query failed:', error);
                    }
                }
                
                if (!tabs || tabs.length === 0) {
                    try {
                        const allTabs = await tabsAPI.query({});
                        const activeTab = allTabs.find(tab => tab.active);
                        if (activeTab) {
                            tabs = [activeTab];
                        }
                    } catch (error) {
                        console.error('CarbonGuru: All tabs query failed:', error);
                    }
                }
                
                if (tabs && tabs.length > 0) {
                    this.currentTab = tabs[0];
                    this.currentUrl = this.currentTab.url;
                    void 0;
                } else {
                    alert('No current tab available - please refresh the page and try again');
                    return;
                }
            } catch (error) {
                console.error('CarbonGuru: Error refreshing current tab:', error);
                alert('Error getting current tab: ' + error.message);
                return;
            }
        }

        try {
            // Show status
            document.getElementById('analysis-status').style.display = 'block';
            document.getElementById('status-content').innerHTML = DOMPurify.sanitize('<div class="loading">Extracting with full page content...</div>');

            void 0;
            void 0;
            
            // Send message to content script for full page extraction
            const tabsAPI = typeof browser !== 'undefined' ? browser.tabs : chrome.tabs;
            const response = await tabsAPI.sendMessage(this.currentTab.id, {
                action: 'extractWithFullPageContent'
            });

            void 0;
            if (response && response.success) {
                this.currentProductData = response.product_data;
                this.showProductDetection(response.product_data);
                document.getElementById('status-content').innerHTML = DOMPurify.sanitize(`
                    <div class="success">
                        ✅ Full page extraction completed successfully!
                    </div>
                `);
            } else {
                document.getElementById('status-content').innerHTML = DOMPurify.sanitize(`
                    <div class="error">
                        ❌ Full page extraction failed: ${response?.error || 'Unknown error'}
                    </div>
                `);
            }
        } catch (error) {
            console.error('CarbonGuru: Error in extractWithFullPage:', error);
            document.getElementById('status-content').innerHTML = DOMPurify.sanitize(`
                <div class="error">
                    ❌ Error: ${error.message}
                </div>
            `);
        }
    }

    showProductDetection(data) {
        void 0;
        
        // Hide current page section and show product detection
        document.getElementById('current-page-section').style.display = 'none';
        document.getElementById('url-input-section').style.display = 'none';
        document.getElementById('product-detection').style.display = 'block';
        
        // Update product info - handle both direct data and preview data
        const productData = data.preview || data; // Handle both formats
        
        // Check if we have clear product name and price information
        const hasProductName = productData.product_name && productData.product_name !== 'Unknown Product' && productData.product_name.trim().length > 0;
        const hasPrice = (productData.price && (productData.price.current_price || productData.price.amount)) || 
                        (typeof productData.price === 'string' && productData.price.trim().length > 0);
        
        const nameElement = document.getElementById('detected-product-name');
        if (nameElement) {
            if (hasProductName) {
                nameElement.textContent = productData.product_name;
            } else {
                nameElement.textContent = 'Product information ready for carbon footprint processing.';
                nameElement.style.fontStyle = 'italic';
                nameElement.style.color = '#666';
            }
        }
        
        const priceElement = document.getElementById('detected-product-price');
        if (priceElement) {
            if (hasPrice) {
                if (productData.price && productData.price.current_price) {
                    priceElement.textContent = `Price: ${productData.price.currency || 'USD'} ${productData.price.current_price}`;
                } else if (productData.price && productData.price.amount) {
                    priceElement.textContent = `Price: ${productData.price.currency || 'USD'} ${productData.price.amount}`;
                } else if (typeof productData.price === 'string') {
                    priceElement.textContent = `Price: ${productData.price}`;
                }
                priceElement.style.fontStyle = 'normal';
                priceElement.style.color = 'inherit';
            } else {
                priceElement.textContent = 'LLM will extract product details from page content.';
                priceElement.style.fontStyle = 'italic';
                priceElement.style.color = '#666';
            }
        }
        
        const categoryElement = document.getElementById('detected-product-category');
        if (categoryElement) {
            if (productData.category && productData.category !== 'Unknown Category') {
                categoryElement.textContent = `Category: ${productData.category}`;
                categoryElement.style.fontStyle = 'normal';
                categoryElement.style.color = 'inherit';
            } else {
                categoryElement.textContent = 'Category: Will be determined during analysis';
                categoryElement.style.fontStyle = 'italic';
                categoryElement.style.color = '#666';
            }
        }
        
        const descriptionElement = document.getElementById('product-description');
        if (descriptionElement) {
            const desc = productData.description || productData.product_description || '';
            if (desc && desc.trim().length > 0) {
                descriptionElement.textContent = `Description: ${desc.substring(0, 100)}${desc.length > 100 ? '...' : ''}`;
                descriptionElement.style.fontStyle = 'normal';
                descriptionElement.style.color = 'inherit';
            } else {
                descriptionElement.textContent = 'Page content captured for intelligent analysis.';
                descriptionElement.style.fontStyle = 'italic';
                descriptionElement.style.color = '#666';
            }
        }
        
        // Store the complete data for later use
        this.currentProductData = productData;
    }

    selectDifferentProduct() {
        void 0;
        // Hide product detection and show current page section
        document.getElementById('product-detection').style.display = 'none';
        document.getElementById('analysis-status').style.display = 'none';
        document.getElementById('results').style.display = 'none';
        document.getElementById('url-input-section').style.display = 'none';
        document.getElementById('current-page-section').style.display = 'block';
    }

    toggleManualUrlInput() {
        void 0;
        const currentPageSection = document.getElementById('current-page-section');
        const urlInputSection = document.getElementById('url-input-section');
        
        if (currentPageSection.style.display === 'none') {
            // Show current page section
            currentPageSection.style.display = 'block';
            urlInputSection.style.display = 'none';
        } else {
            // Show manual URL input
            currentPageSection.style.display = 'none';
            urlInputSection.style.display = 'block';
        }
    }

    async checkBackgroundScript() {
        try {
            void 0;
            const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
            await runtime.sendMessage({ action: 'ping' });
            void 0;
            return true;
        } catch (error) {
            console.error('CarbonGuru: Background script not available:', error);
            return false;
        }
    }

    // ==========================================================================
    // Impact Section (V3 horizontal row: chip | offset | removal)
    // ==========================================================================

    /**
     * 12-step color spectrum for impact percentage (red → yellow → green)
     * Matches header-impact-widget/render.php lines 93-106
     */
    static IMPACT_COLOR_STEPS = [
        { min: 0,  max: 8,   bg: '#9B1B1B', dk: '#7A1515', bd: '#FECACA' },
        { min: 9,  max: 16,  bg: '#B92525', dk: '#961E1E', bd: '#FDD0D0' },
        { min: 17, max: 25,  bg: '#D43030', dk: '#AE2727', bd: '#FBD5D5' },
        { min: 26, max: 33,  bg: '#E04545', dk: '#C13535', bd: '#FCDADA' },
        { min: 34, max: 42,  bg: '#B8860B', dk: '#956D09', bd: '#F5E0A0' },
        { min: 43, max: 50,  bg: '#C49B12', dk: '#A1800F', bd: '#F7E8A8' },
        { min: 51, max: 58,  bg: '#B5A31A', dk: '#928416', bd: '#F3ECB0' },
        { min: 59, max: 66,  bg: '#8DA11E', dk: '#738418', bd: '#E0EFA0' },
        { min: 67, max: 75,  bg: '#5A8C1E', dk: '#4A7318', bd: '#CBE4A0' },
        { min: 76, max: 83,  bg: '#378225', dk: '#2D6B1E', bd: '#B0DCA5' },
        { min: 84, max: 91,  bg: '#0A9944', dk: '#087D38', bd: '#8DD4AC' },
        { min: 92, max: 100, bg: '#067F38', dk: '#056830', bd: '#8DD4AC' },
    ];

    /**
     * CG tree SVG path data
     */
    static TREE_PATH = 'M57.7,53c-0.1,2-1.4,4-1.4,4c0-0.7-0.7-3-0.7-3c-0.8,0.3-2.3,0.3-2.3,0.3s0.5,1.3,0.5,2.9c0,1.6-2.1,7.9-2.1,7.9c-2.9-5.1-2.4-9.5-2.4-9.5c-4.4,0.1-8.1-1.1-8.1-1.1l0.9,3.4c-4.2-1.6-4.3-4.2-4.3-4.2l-2.2-0.6c0.4,1.9,3.5,5.5,4.8,6.6c1.4,1.1,4.2,4,5.9,6.1c1.7,2.1,1.9,3.6,2.1,5.2c0.2,1.6-0.8,7.4-0.8,8.4c0,1-1.5,9.4-2.2,10.7c-0.7,1.3-3.5,2.6-3.5,2.6s2.1,0.7,4.3,0.7c2.2,0,7.6-1.1,9-1.1c1.4,0,2,0.4,3.7,0.4c1.7,0,2.1-0.4,2.1-0.4c-4.7-1.7-4.7-2.3-4.7-2.3c-1.5-2.7-2.2-9.9-1.3-19.1c0.9-9.2,7.7-14.8,7.7-14.8C59.3,54.9,57.7,53,57.7,53z M86.8,33.3c4.5-20.8-13.9-19-13.9-19C65.6,1.9,46.1,6.9,46.1,6.9c-21.1-9.3-22.8,7.1-22.8,7.1c-17.2-1.3-9.6,16.3-9.6,16.3c-9.5,16.1,4.6,15.6,4.6,15.6c1.9,12.7,16.6,5.2,16.6,5.2C48.4,56.3,58.5,50.9,58.5,50.9c13.7,11.4,20.2,0,20.2,0C94.4,51.6,86.8,33.3,86.8,33.3z';

    /**
     * Get the color step for a given percentage
     */
    getImpactColorStep(percent) {
        const steps = CarbonGuruPopup.IMPACT_COLOR_STEPS;
        for (let i = steps.length - 1; i >= 0; i--) {
            if (percent >= steps[i].min) {
                return steps[i];
            }
        }
        return steps[0];
    }

    /**
     * Format a number with commas (e.g. 4515.8 → "4,515.8")
     */
    formatNumber(num, decimals = 1) {
        return Number(num).toLocaleString('en-US', {
            minimumFractionDigits: decimals,
            maximumFractionDigits: decimals,
        });
    }

    /**
     * Load impact section data from background and render
     */
    async loadImpactSection() {
        try {
            void 0;
            const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
            const data = await runtime.sendMessage({ action: 'getUserImpact' });

            if (!data || !data.logged_in) {
                void 0;
                return;
            }

            // Don't show if no offset/removal activity
            if (data.totals.kg_offset <= 0 && data.totals.kg_removed <= 0) {
                void 0;
                return;
            }

            this.renderImpactSection(data);
        } catch (error) {
            void 0;
        }
    }

    /**
     * Build and render the V3 impact section (chip | offset | removal)
     */
    renderImpactSection(data) {
        const section = document.getElementById('impact-section');
        const row = document.getElementById('impact-row');
        const dashboardLink = document.getElementById('impact-dashboard-link');

        if (!section || !row) return;

        // Set dashboard link
        if (dashboardLink && this.environmentConfig) {
            dashboardLink.href = `${this.environmentConfig.backendUrl}/carbon-dashboard/`;
        }

        const pct = data.footprint_percent_addressed || 0;
        const step = this.getImpactColorStep(pct);
        const clipY = 100 - pct;
        const clipH = pct;
        const clipId = 'cg-ext-clip-' + Date.now();
        const SVG_NS = 'http://www.w3.org/2000/svg';

        // Clear previous content
        row.innerHTML = '';

        // --- Column 1: Impact % Chip ---
        const chip = document.createElement('div');
        chip.className = 'impact-section__chip';
        chip.style.border = `2px solid ${step.bd}`;

        const chipTop = document.createElement('div');
        chipTop.className = 'impact-section__chip-top';

        const chipTree = document.createElement('div');
        chipTree.className = 'impact-section__chip-tree';

        const treeSvg = document.createElementNS(SVG_NS, 'svg');
        treeSvg.setAttribute('width', '28');
        treeSvg.setAttribute('height', '34');
        treeSvg.setAttribute('viewBox', '0 0 100 100');
        const defs = document.createElementNS(SVG_NS, 'defs');
        const clipPathEl = document.createElementNS(SVG_NS, 'clipPath');
        clipPathEl.setAttribute('id', clipId);
        const clipRect = document.createElementNS(SVG_NS, 'rect');
        clipRect.setAttribute('x', '0');
        clipRect.setAttribute('y', String(clipY));
        clipRect.setAttribute('width', '100');
        clipRect.setAttribute('height', String(clipH));
        clipPathEl.appendChild(clipRect);
        defs.appendChild(clipPathEl);
        treeSvg.appendChild(defs);
        const outlinePath = document.createElementNS(SVG_NS, 'path');
        outlinePath.setAttribute('d', CarbonGuruPopup.TREE_PATH);
        outlinePath.setAttribute('fill', 'none');
        outlinePath.setAttribute('stroke', step.bg);
        outlinePath.setAttribute('stroke-width', '2.5');
        treeSvg.appendChild(outlinePath);
        const fillPath = document.createElementNS(SVG_NS, 'path');
        fillPath.setAttribute('d', CarbonGuruPopup.TREE_PATH);
        fillPath.setAttribute('fill', step.bg);
        fillPath.setAttribute('clip-path', `url(#${clipId})`);
        treeSvg.appendChild(fillPath);
        chipTree.appendChild(treeSvg);
        chipTop.appendChild(chipTree);

        const chipVal = document.createElement('span');
        chipVal.className = 'impact-section__chip-val';
        chipVal.style.color = step.dk;
        chipVal.textContent = String(pct);
        chipTop.appendChild(chipVal);

        chip.appendChild(chipTop);

        const chipLabel = document.createElement('span');
        chipLabel.className = 'impact-section__chip-label';
        chipLabel.style.color = step.bg;
        chipLabel.textContent = 'Impact %';
        chip.appendChild(chipLabel);

        row.appendChild(chip);

        // --- Column 2: Offset stat ---
        const offsetStat = document.createElement('div');
        offsetStat.className = 'impact-section__stat impact-section__stat--offset';
        const offsetVal = document.createElement('div');
        offsetVal.className = 'impact-section__stat-val';
        offsetVal.textContent = this.formatNumber(data.totals.kg_offset);
        offsetStat.appendChild(offsetVal);
        const offsetLabel = document.createElement('div');
        offsetLabel.className = 'impact-section__stat-label';
        offsetLabel.textContent = 'kg CO\u2082e Offset';
        offsetStat.appendChild(offsetLabel);
        row.appendChild(offsetStat);

        // --- Column 3: Removal stat ---
        const removalStat = document.createElement('div');
        removalStat.className = 'impact-section__stat impact-section__stat--removal';
        const removalVal = document.createElement('div');
        removalVal.className = 'impact-section__stat-val';
        removalVal.textContent = this.formatNumber(data.totals.kg_removed);
        removalStat.appendChild(removalVal);
        const removalLabel = document.createElement('div');
        removalLabel.className = 'impact-section__stat-label';
        removalLabel.textContent = 'kg CO\u2082e Removed';
        removalStat.appendChild(removalLabel);
        row.appendChild(removalStat);

        // --- Recent Orders (collapsible) ---
        const orders = data.recent_orders || [];
        const ordersContainer = document.getElementById('impact-orders');
        if (ordersContainer && orders.length > 0) {
            ordersContainer.innerHTML = '';

            // Toggle bar
            const toggle = document.createElement('div');
            toggle.className = 'impact-orders__toggle';
            toggle.addEventListener('click', () => {
                toggle.classList.toggle('open');
                list.classList.toggle('open');
            });

            const toggleText = document.createElement('span');
            toggleText.className = 'impact-orders__toggle-text';
            toggleText.textContent = 'Recent Orders';
            toggle.appendChild(toggleText);

            const SVG_ARROW_NS = 'http://www.w3.org/2000/svg';
            const arrowSvg = document.createElementNS(SVG_ARROW_NS, 'svg');
            arrowSvg.setAttribute('class', 'impact-orders__toggle-arrow');
            arrowSvg.setAttribute('viewBox', '0 0 10 10');
            const arrowPath = document.createElementNS(SVG_ARROW_NS, 'path');
            arrowPath.setAttribute('d', 'M2 3.5L5 6.5L8 3.5');
            arrowPath.setAttribute('fill', 'none');
            arrowPath.setAttribute('stroke', 'currentColor');
            arrowPath.setAttribute('stroke-width', '1.5');
            arrowPath.setAttribute('stroke-linecap', 'round');
            arrowPath.setAttribute('stroke-linejoin', 'round');
            arrowSvg.appendChild(arrowPath);
            toggle.appendChild(arrowSvg);

            ordersContainer.appendChild(toggle);

            // Orders list
            const list = document.createElement('div');
            list.className = 'impact-orders__list';

            for (const order of orders) {
                const orderType = order.type || 'offset';

                const orderEl = document.createElement('div');
                orderEl.className = 'impact-order';

                // Color bar
                const bar = document.createElement('div');
                bar.className = `impact-order__bar impact-order__bar--${orderType}`;
                orderEl.appendChild(bar);

                // Info (id + date)
                const info = document.createElement('div');
                info.className = 'impact-order__info';
                const idEl = document.createElement('div');
                idEl.className = 'impact-order__id';
                idEl.textContent = `#${order.order_id}`;
                info.appendChild(idEl);
                const dateEl = document.createElement('div');
                dateEl.className = 'impact-order__date';
                dateEl.textContent = order.date || '';
                info.appendChild(dateEl);
                orderEl.appendChild(info);

                // Badge
                const badge = document.createElement('span');
                badge.className = `impact-order__badge impact-order__badge--${orderType}`;
                badge.textContent = orderType === 'offset' ? 'Offset' : 'Removal';
                orderEl.appendChild(badge);

                // Values
                const val = document.createElement('div');
                val.className = 'impact-order__val';
                const kgEl = document.createElement('div');
                kgEl.className = `impact-order__kg impact-order__kg--${orderType}`;
                kgEl.textContent = `${this.formatNumber(order.kg_co2e)} kg`;
                val.appendChild(kgEl);
                const costEl = document.createElement('div');
                costEl.className = 'impact-order__cost';
                costEl.textContent = `$${this.formatNumber(order.cost, 2)}`;
                val.appendChild(costEl);
                orderEl.appendChild(val);

                list.appendChild(orderEl);
            }

            ordersContainer.appendChild(list);
            ordersContainer.style.display = '';
        }

        // Show the section and mark as loaded for minimize/restore
        section.style.display = '';
        section.dataset.loaded = 'true';
        void 0;
    }

    async loadRecentItems() {
        try {
            void 0;
            void 0;
            const listElement = document.getElementById('recent-items-list');
            void 0;

            if (!listElement) {
                console.error('CarbonGuru: CRITICAL - recent-items-list element not found in DOM!');
                return;
            }

            void 0;
            const backgroundAvailable = await this.checkBackgroundScript();
            void 0;

            if (!backgroundAvailable) {
                listElement.innerHTML = DOMPurify.sanitize('<div class="error">Background script not available</div>');
                return;
            }

            void 0;
            const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
            const response = await runtime.sendMessage({
                action: 'getRecentItems',
                limit: 10
            });

            void 0;

            if (response && response.items && response.items.length > 0) {
                void 0;
                this.displayRecentItems(response.items);
                void 0;
            } else {
                void 0;
                listElement.innerHTML = DOMPurify.sanitize('<div class="no-items">No recent items found</div>');
            }
            void 0;
        } catch (error) {
            console.error('CarbonGuru: Error loading recent items:', error);
            console.error('CarbonGuru: Error stack:', error.stack);
            const listElement = document.getElementById('recent-items-list');
            if (listElement) {
                listElement.innerHTML = DOMPurify.sanitize('<div class="error">Error loading recent items: ' + error.message + '</div>');
            }
        }
    }

    displayRecentItems(items) {
        void 0;
        void 0;

        const container = document.getElementById('recent-items-list');
        void 0;

        if (!items || items.length === 0) {
            void 0;
            if (container) {
                container.innerHTML = DOMPurify.sanitize('<div class="no-items">No recent items found</div>');
            }
            return;
        }

        // Debug: Log item structure to see available IDs
        void 0;

        const itemsHtml = items.slice(0, 10).map((item, index) => {
            const summaryTypes = ['cart', 'order', 'email_cart', 'email_order', 'email'];
            const isCart = summaryTypes.includes(item.item_type) ||
                item.analysis_type === 'shopping_cart' ||
                item.analysis_type === 'order_page' ||
                item.type === 'cart' || item.type === 'order';
            const itemIcon = isCart ? '🛒' : '';
            const itemClass = isCart ? 'recent-item cart-item' : 'recent-item';

            // For cart/order summaries, parse the enhanced title to get item count and total
            let displayName = item.product_name || 'Unknown Product';
            let footprintText = `${item.product_carbon_footprint || 'N/A'} kg CO₂e`;

            // Add info icon if item has PCF comparison
            const infoIcon = item.pcf_comparison ? ' <span class="carbon-info-icon recent-info-icon" data-item-index="' + index + '" title="View comparable product">ⓘ</span>' : '';

            // DISABLED: Offset cost badge - costs are now shown in buttons
            // const offsetCostValue = item.offset_cost_usd ?? item.display_cost_usd ?? item.orderable_cost_usd;
            // const offsetCostBadge = offsetCostValue && offsetCostValue > 0
            //     ? `<span class="carbon-cost-label">Offset Cost: $${parseFloat(offsetCostValue).toFixed(2)} USD</span>`
            //     : '';
            const offsetCostBadge = ''; // Disabled - costs shown in buttons
            // DISABLED: Social cost of carbon display - uncomment to re-enable
            // const carbonCostBadge = item.social_cost_usd && item.social_cost_usd > 0
            //     ? `<span class="carbon-cost-label">Carbon Cost: $${parseFloat(item.social_cost_usd).toFixed(2)} USD</span>`
            //     : '';
            const carbonCostBadge = ''; // Disabled - social cost not shown

            if (isCart) {
                // Format cart footprint to one decimal place
                const formattedFootprint = item.product_carbon_footprint
                    ? parseFloat(item.product_carbon_footprint).toFixed(1)
                    : 'N/A';
                // Include info icon for carts if they have PCF comparison
                footprintText = `${formattedFootprint} kg CO₂e${infoIcon}`;
            } else {
                // Format individual item footprint to one decimal place
                const formattedFootprint = item.product_carbon_footprint
                    ? parseFloat(item.product_carbon_footprint).toFixed(1)
                    : 'N/A';
                // Include info icon for individual items if they have PCF comparison
                footprintText = `${formattedFootprint} kg CO₂e${infoIcon}`;
            }

            // Build action buttons based on item type - dual offset/removal buttons
            // Use pre-stored costs if available, otherwise show '...' and fetch later
            let actionButtons = '';
            const buttonClass = isCart ? 'add-cart-to-basket-btn' : 'add-to-basket-btn';
            const offsetCostDisplay = item.offset_cost_usd ? formatCost(item.offset_cost_usd) : '...';
            const removalCostDisplay = item.removal_cost_usd ? formatCost(item.removal_cost_usd) : '...';
            actionButtons = `
                <div class="btn-group">
                    <button class="btn-small offset-btn ${buttonClass}" data-item-index="${index}" data-item-id="${item.carbonguru_id || item.id}" data-preference-type="offset">
                        <span class="btn-label">Offset:</span>
                        <span class="btn-cost">${offsetCostDisplay}</span>
                    </button>
                    <button class="btn-small removal-btn ${buttonClass}" data-item-index="${index}" data-item-id="${item.carbonguru_id || item.id}" data-preference-type="removal">
                        <span class="btn-label">Remove:</span>
                        <span class="btn-cost">${removalCostDisplay}</span>
                    </button>
                </div>
            `;

            // Don't show date - cart summaries have date in title, individual items don't need it
            // Item name is now a clickable link (View Details button removed)
            return `
                <div class="${itemClass}" data-item-index="${index}">
                    <a href="#" class="item-name item-name-link" data-item-id="${item.carbonguru_id || item.id}">${itemIcon}${itemIcon ? ' ' : ''}${displayName}</a>
                    <div class="item-footprint">
                        <span>${footprintText}</span>
                        ${offsetCostBadge}
                        ${carbonCostBadge}
                    </div>
                    <div class="item-actions">
                        ${actionButtons}
                    </div>
                </div>
            `;
        }).join('');

        container.innerHTML = DOMPurify.sanitize(itemsHtml);

        // Store items for basket operations
        this.currentRecentItems = items;

        // Fetch and display costs for all items
        this.fetchRecentItemCosts(items.slice(0, 10));
        
        // Add event listeners for dual offset/removal buttons (both individual items and carts)
        const allBasketButtons = container.querySelectorAll('.offset-btn, .removal-btn');
        allBasketButtons.forEach(button => {
            button.addEventListener('click', async (e) => {
                e.preventDefault();
                e.stopPropagation();

                const preferenceType = button.getAttribute('data-preference-type'); // 'offset' or 'removal'
                const isCart = button.classList.contains('add-cart-to-basket-btn');
                const costSpan = button.querySelector('.btn-cost');
                const originalCost = costSpan ? costSpan.textContent : '';

                let itemData;
                try {
                    const itemIndex = parseInt(button.getAttribute('data-item-index'));
                    if (isNaN(itemIndex) || !this.currentRecentItems || !this.currentRecentItems[itemIndex]) {
                        throw new Error('Invalid item index or no item data');
                    }
                    itemData = this.currentRecentItems[itemIndex];
                } catch (parseError) {
                    console.error('CarbonGuru: Error getting item data:', parseError);
                    if (costSpan) costSpan.textContent = 'Error';
                    button.classList.add('error');
                    setTimeout(() => {
                        if (costSpan) costSpan.textContent = originalCost;
                        button.classList.remove('error');
                        button.disabled = false;
                    }, 2000);
                    return;
                }

                void 0;

                try {
                    // Disable button during request
                    button.disabled = true;
                    if (costSpan) costSpan.textContent = '...';

                    const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;

                    if (isCart && itemData.items && Array.isArray(itemData.items)) {
                        // Handle cart - add all items with preference type
                        let successCount = 0;
                        const totalItems = itemData.items.length;

                        for (const item of itemData.items) {
                            const basketItem = {
                                carbonguru_id: item.carbonguru_id || item.id,
                                product_name: item.product_name || item.name || 'Unknown Product',
                                product_carbon_footprint: item.product_carbon_footprint || item.carbon_footprint,
                                details_url: item.details_url || ''
                            };

                            const response = await runtime.sendMessage({
                                action: 'addToBasketWithType',
                                item: basketItem,
                                preferenceType: preferenceType
                            });

                            if (response && response.success) {
                                successCount++;
                            }
                        }

                        if (successCount === totalItems) {
                            if (costSpan) costSpan.textContent = '✓';
                            button.classList.add('basket-added');
                            void 0;
                        } else if (successCount > 0) {
                            if (costSpan) costSpan.textContent = `${successCount}/${totalItems}`;
                            button.classList.add('partial-success');
                        } else {
                            if (costSpan) costSpan.textContent = 'Error';
                            button.classList.add('error');
                        }
                    } else {
                        // Handle individual item
                        // Get the fee-included cost from the button (set by fetchRecentItemCosts)
                        const costText = originalCost.replace('$', '').trim();
                        const feeIncludedCost = parseFloat(costText) || 0;

                        const response = await runtime.sendMessage({
                            action: 'addToBasketWithType',
                            item: {
                                carbonguru_id: itemData.carbonguru_id || itemData.id,
                                product_name: itemData.product_name,
                                product_carbon_footprint: itemData.product_carbon_footprint,
                                details_url: itemData.details_url || '',
                                fee_included_cost: feeIncludedCost  // Pass the fee-included cost
                            },
                            preferenceType: preferenceType
                        });

                        if (response && response.success) {
                            if (costSpan) costSpan.textContent = '✓';
                            button.classList.add('basket-added');
                            void 0;
                        } else {
                            if (costSpan) costSpan.textContent = 'Error';
                            button.classList.add('error');
                            console.error('CarbonGuru: Failed to add item:', response);
                        }
                    }

                    // Refresh basket display
                    this.refreshBasketDisplay();

                    // Reset button after 2 seconds
                    setTimeout(() => {
                        if (costSpan) costSpan.textContent = originalCost;
                        button.classList.remove('basket-added', 'partial-success', 'error');
                        button.disabled = false;
                    }, 2000);

                } catch (error) {
                    console.error('CarbonGuru: Error adding to basket:', error);
                    if (costSpan) costSpan.textContent = 'Error';
                    button.classList.add('error');
                    button.disabled = false;

                    setTimeout(() => {
                        if (costSpan) costSpan.textContent = originalCost;
                        button.classList.remove('error');
                    }, 2000);
                }
            });
        });

        // Add event listeners for clickable item name links
        const itemNameLinks = container.querySelectorAll('.item-name-link');
        itemNameLinks.forEach(link => {
            link.addEventListener('click', (e) => {
                e.preventDefault();
                const itemId = link.getAttribute('data-item-id');
                this.viewItemDetails(itemId);
            });
        });

        // Setup PCF comparison overlay for info icons
        this.setupRecentItemsPcfOverlay();

        void 0;
        void 0;
        void 0;
    }

    setupRecentItemsPcfOverlay() {
        // Add event listeners to info icons in Recent Items for PCF comparison overlay
        const infoIcons = document.querySelectorAll('.recent-info-icon');

        infoIcons.forEach(icon => {
            const itemIndex = parseInt(icon.getAttribute('data-item-index'));
            const itemData = this.currentRecentItems?.[itemIndex];

            if (itemData && itemData.pcf_comparison) {
                // Click to show overlay
                icon.addEventListener('click', (e) => {
                    e.stopPropagation();
                    this.showPcfOverlay(itemData, icon);
                });

                // Hover to show overlay
                icon.addEventListener('mouseenter', () => {
                    this.showPcfOverlay(itemData, icon);
                });
            }
        });
    }

    showPcfOverlay(item, triggeringIcon) {
        // Remove existing overlay
        const existingOverlay = document.getElementById('carbon-pcf-overlay');
        if (existingOverlay) existingOverlay.remove();

        const match = item.pcf_comparison.match;
        const descriptionHtml = match.description ? ` <span class="carbon-comparison-description">(${this.escapeHtml(match.description)})</span>` : '';
        const overlay = document.createElement('div');
        overlay.id = 'carbon-pcf-overlay';
        overlay.className = 'carbon-pcf-overlay-positioned';
        overlay.innerHTML = DOMPurify.sanitize(`
            <div class="carbon-pcf-overlay-content">
                <div class="carbon-comparison-header">Similar footprint to:</div>
                <div class="carbon-comparison-content">
                    <div class="carbon-comparison-product">
                        <div class="carbon-comparison-name">${this.escapeHtml(match.base_product_name)}${descriptionHtml}</div>
                        <div class="carbon-comparison-pcf">${match.base_pcf.toFixed(2)} kg CO₂e each</div>
                    </div>
                    <!-- DISABLED: Multiplier display disabled while using base products only -->
                    <!-- <div class="carbon-comparison-multiplier">${match.multiplier}x</div> -->
                </div>
                <div class="carbon-comparison-match">
                    <!-- DISABLED: Total footprint redundant when showing base products only -->
                    <!-- <span class="carbon-comparison-total">${match.total_pcf.toFixed(2)} kg CO₂e</span> -->
                    <span class="carbon-comparison-accuracy">${(match.match_score * 100).toFixed(1)}% match</span>
                </div>
            </div>
        `);

        document.body.appendChild(overlay);

        // Position overlay centered in popup window, above the icon
        if (triggeringIcon) {
            const iconRect = triggeringIcon.getBoundingClientRect();
            const overlayContent = overlay.querySelector('.carbon-pcf-overlay-content');

            // Wait for overlay to be added to DOM to get its dimensions
            setTimeout(() => {
                const overlayRect = overlayContent.getBoundingClientRect();

                // Center horizontally within the popup window
                const left = (window.innerWidth - overlayRect.width) / 2;

                // Position above the icon with 8px gap
                const bottom = window.innerHeight - iconRect.top + 8;

                overlay.style.position = 'fixed';
                overlay.style.left = `${left}px`;
                overlay.style.bottom = `${bottom}px`;
                overlay.style.background = 'transparent';
                overlay.style.pointerEvents = 'auto';

                // Allow content to be interactive
                overlayContent.style.pointerEvents = 'auto';
            }, 0);
        }

        // Close on click outside or Escape key
        const closeOverlay = () => {
            overlay.remove();
        };

        overlay.addEventListener('click', (e) => {
            if (e.target === overlay) closeOverlay();
        });

        overlay.addEventListener('mouseleave', closeOverlay);

        document.addEventListener('keydown', function escHandler(e) {
            if (e.key === 'Escape') {
                closeOverlay();
                document.removeEventListener('keydown', escHandler);
            }
        });

        // Auto-close after 5 seconds
        setTimeout(closeOverlay, 5000);
    }

    escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }

    async viewItemDetails(itemId) {
        try {
            const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
            const response = await runtime.sendMessage({
                action: 'getCarbonFootprint',
                itemId: itemId
            });

            void 0;

            if (response && response.success && response.item) {
                const tabsAPI = typeof browser !== 'undefined' ? browser.tabs : chrome.tabs;

                // Use details_url from API response (already contains WooCommerce URL if available)
                if (response.item.details_url) {
                    void 0;
                    // details_url might be a full URL or a relative path, let getProductUrl handle it
                    const detailsUrl = await this.getProductUrl(response.item.details_url);
                    tabsAPI.create({ url: detailsUrl });
                } else {
                    // Fallback to constructing path (shouldn't happen with API)
                    const productPath = `/product/${response.item.session_id}/${response.item.carbonguru_id}`;
                    const detailsUrl = await this.getProductUrl(productPath);
                    tabsAPI.create({ url: detailsUrl });
                }
            } else {
                alert('Error loading item details: ' + (response?.error || 'Unknown error'));
            }
        } catch (error) {
            console.error('CarbonGuru: Error viewing item details:', error);
            alert('Error loading item details: ' + error.message);
        }
    }

    // Offset Basket Management Methods
    async loadBasket() {
        try {
            void 0;
            const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
            const response = await runtime.sendMessage({
                action: 'getBasket'
            });

            if (response && response.success && response.basket) {
                void 0;
                if (response.basket.rawCart) {
                    void 0;
                }
                this.displayBasket(response.basket);
                // Update sync status whenever basket is loaded
                await this.updateSyncStatus();
            } else {
                console.error('CarbonGuru: Failed to load basket:', response);
                const backendUrl = this.environmentConfig?.backendUrl || 'https://carbonguru.io';
                this.displayBasket({
                    items: [],
                    count: 0,
                    totalFootprint: 0,
                    totalFootprintFormatted: '0 kg CO₂e',
                    totalCarbonCost: 0,
                    totalCarbonCostFormatted: null,
                    cart_url: `${backendUrl}/cart/`,
                    checkout_url: `${backendUrl}/checkout/`
                });
            }
        } catch (error) {
            console.error('CarbonGuru: Error loading basket:', error);
            const backendUrl = this.environmentConfig?.backendUrl || 'https://carbonguru.io';
            this.displayBasket({
                items: [],
                count: 0,
                totalFootprint: 0,
                totalFootprintFormatted: '0 kg CO₂e',
                totalCarbonCost: 0,
                totalCarbonCostFormatted: null,
                cart_url: `${backendUrl}/cart/`,
                checkout_url: `${backendUrl}/checkout/`
            });
        }
    }

    displayBasket(basket) {
        const basketSection = document.getElementById('offset-basket');
        const basketItemsList = document.getElementById('basket-items-list');
        const basketTotalTop = document.getElementById('basket-total-top');
        const basketTotalBottom = document.getElementById('basket-total-bottom');

        if (!basketSection || !basketItemsList) return;

        // Update totals with separate offset and removal cost badges
        const offsetTotalValue = basket.totalOffsetCost !== undefined && basket.totalOffsetCost !== null
            ? parseFloat(basket.totalOffsetCost)
            : 0;
        const removalTotalValue = basket.totalRemovalCost !== undefined && basket.totalRemovalCost !== null
            ? parseFloat(basket.totalRemovalCost)
            : 0;

        // Hide total carbon footprint display (moved to group headers)
        if (basketTotalTop) basketTotalTop.style.display = 'none';
        if (basketTotalBottom) basketTotalBottom.style.display = 'none';

        // Always show basket section
        basketSection.style.display = 'block';

        // Separate items by type
        const offsetItems = basket.items ? basket.items.filter(item => item.preference_type !== 'removal') : [];
        const removalItems = basket.items ? basket.items.filter(item => item.preference_type === 'removal') : [];

        // Calculate group totals
        const offsetCarbon = offsetItems.reduce((sum, item) => sum + (item.product_carbon_footprint * (item.quantity || 1)), 0);
        const removalCarbon = removalItems.reduce((sum, item) => sum + (item.product_carbon_footprint * (item.quantity || 1)), 0);

        // CG Tree icon SVG (inline for use in headers)
        const cgTreeIcon = `<svg class="cart-group-tree-icon" viewBox="0 0 100 100" width="16" height="16"><path fill="currentColor" d="M57.7,53c-0.1,2-1.4,4-1.4,4c0-0.7-0.7-3-0.7-3c-0.8,0.3-2.3,0.3-2.3,0.3s0.5,1.3,0.5,2.9c0,1.6-2.1,7.9-2.1,7.9c-2.9-5.1-2.4-9.5-2.4-9.5c-4.4,0.1-8.1-1.1-8.1-1.1l0.9,3.4c-4.2-1.6-4.3-4.2-4.3-4.2l-2.2-0.6c0.4,1.9,3.5,5.5,4.8,6.6c1.4,1.1,4.2,4,5.9,6.1c1.7,2.1,1.9,3.6,2.1,5.2c0.2,1.6-0.8,7.4-0.8,8.4c0,1-1.5,9.4-2.2,10.7c-0.7,1.3-3.5,2.6-3.5,2.6s2.1,0.7,4.3,0.7c2.2,0,7.6-1.1,9-1.1c1.4,0,2,0.4,3.7,0.4c1.7,0,2.1-0.4,2.1-0.4c-4.7-1.7-4.7-2.3-4.7-2.3c-1.5-2.7-2.2-9.9-1.3-19.1c0.9-9.2,7.7-14.8,7.7-14.8C59.3,54.9,57.7,53,57.7,53z M86.8,33.3c4.5-20.8-13.9-19-13.9-19C65.6,1.9,46.1,6.9,46.1,6.9c-21.1-9.3-22.8,7.1-22.8,7.1c-17.2-1.3-9.6,16.3-9.6,16.3c-9.5,16.1,4.6,15.6,4.6,15.6c1.9,12.7,16.6,5.2,16.6,5.2C48.4,56.3,58.5,50.9,58.5,50.9c13.7,11.4,20.2,0,20.2,0C94.4,51.6,86.8,33.3,86.8,33.3z"/></svg>`;

        // Display basket contents based on items
        if (basket.count > 0) {
            let itemsHtml = '';

            // Render Offsets group if there are offset items
            if (offsetItems.length > 0) {
                const offsetTotalDisplay = offsetTotalValue > 0 ? formatCost(offsetTotalValue) : '';
                itemsHtml += `
                    <div class="cart-group-header offset-group">
                        ${cgTreeIcon}
                        <span class="cart-group-label">Offsets</span>
                        <span class="group-count">${offsetItems.length} item${offsetItems.length !== 1 ? 's' : ''} · ${this.formatFootprint(offsetCarbon)}${offsetTotalDisplay ? ` · ${offsetTotalDisplay}` : ''}</span>
                    </div>
                `;
                itemsHtml += offsetItems.map(item => this.renderCartItem(item, 'offset')).join('');
            }

            // Render Removals group if there are removal items
            if (removalItems.length > 0) {
                const removalTotalDisplay = removalTotalValue > 0 ? formatCost(removalTotalValue) : '';
                itemsHtml += `
                    <div class="cart-group-header removal-group">
                        ${cgTreeIcon}
                        <span class="cart-group-label">Removals</span>
                        <span class="group-count">${removalItems.length} item${removalItems.length !== 1 ? 's' : ''} · ${this.formatFootprint(removalCarbon)}${removalTotalDisplay ? ` · ${removalTotalDisplay}` : ''}</span>
                    </div>
                `;
                itemsHtml += removalItems.map(item => this.renderCartItem(item, 'removal')).join('');
            }

            basketItemsList.innerHTML = DOMPurify.sanitize(itemsHtml);

            // Add event listeners for remove buttons
            this.setupBasketRemoveListeners();

        } else {
            basketItemsList.innerHTML = DOMPurify.sanitize(`
                <div class="basket-empty">
                    <div style="margin-bottom: 8px;">Your carbon cart is empty.</div>
                    <div style="font-size: 11px; color: #666; margin-bottom: 12px;">
                        Add items from Recent Items to offset or remove their carbon footprint.
                    </div>
                    <a href="${basket.cart_url || '#'}" target="_blank"
                       style="color: var(--cg-green-600); text-decoration: underline; font-size: 12px;">
                        View Details
                    </a>
                </div>
            `);
        }

        // Show/hide type-specific button groups based on cart contents
        const offsetButtons = document.getElementById('offset-buttons');
        const removalButtons = document.getElementById('removal-buttons');

        // Update item counts and costs for each cart type (used by footer buttons)
        const offsetItemCountEl = document.getElementById('offset-item-count');
        const offsetTotalCostEl = document.getElementById('offset-total-cost');
        const removalItemCountEl = document.getElementById('removal-item-count');
        const removalTotalCostEl = document.getElementById('removal-total-cost');

        if (offsetItemCountEl) {
            offsetItemCountEl.textContent = offsetItems.length > 0 ? `(${offsetItems.length} item${offsetItems.length !== 1 ? 's' : ''})` : '';
        }
        if (offsetTotalCostEl) {
            offsetTotalCostEl.textContent = offsetTotalValue > 0 ? formatCost(offsetTotalValue) : '';
        }
        if (removalItemCountEl) {
            removalItemCountEl.textContent = removalItems.length > 0 ? `(${removalItems.length} item${removalItems.length !== 1 ? 's' : ''})` : '';
        }
        if (removalTotalCostEl) {
            removalTotalCostEl.textContent = removalTotalValue > 0 ? formatCost(removalTotalValue) : '';
        }

        // Show offset buttons if there are offset items OR offset cost
        if (offsetButtons) {
            offsetButtons.style.display = (offsetItems.length > 0 || offsetTotalValue > 0) ? 'flex' : 'none';
        }

        // Show removal buttons if there are removal items OR removal cost
        if (removalButtons) {
            removalButtons.style.display = (removalItems.length > 0 || removalTotalValue > 0) ? 'flex' : 'none';
        }

        // Fetch costs for basket items that are missing prices (async, updates DOM)
        if (basket.items && basket.items.length > 0) {
            this.fetchBasketItemCosts(basket.items);
        }
    }

    /**
     * Render a single cart item with type-specific styling and cost badge
     */
    renderCartItem(item, type) {
        const quantity = item.quantity || 1;
        const totalFootprint = item.product_carbon_footprint * quantity;

        // Get the cost for this item based on its type
        let itemCost = 0;
        if (type === 'removal') {
            itemCost = item.removal_cost_usd ? parseFloat(item.removal_cost_usd) * quantity : 0;
        } else {
            const offsetCostValue = item.offset_cost_usd ?? item.display_cost_usd ?? item.orderable_cost_usd;
            itemCost = offsetCostValue ? parseFloat(offsetCostValue) * quantity : 0;
        }

        const quantityBadge = quantity > 1 ? `<span class="quantity-badge">×${quantity}</span>` : '';
        const typeClass = type === 'removal' ? 'removal-item' : 'offset-item';
        const costClass = type === 'removal' ? 'removal-cost' : 'offset-cost';
        const costBadge = itemCost > 0 ? `<span class="cost ${costClass}">${formatCost(itemCost)}</span>` : '';

        return `
            <div class="basket-item ${typeClass}" data-cart-item-key="${item.cart_item_key}">
                <div class="basket-item-info">
                    <div class="basket-item-name">
                        ${item.product_name}
                        ${quantityBadge}
                    </div>
                    <div class="basket-item-footprint">
                        <span class="carbon">${this.formatFootprint(totalFootprint)}</span>
                        ${costBadge}
                    </div>
                </div>
                <button class="basket-remove-btn"
                        data-cart-item-key="${item.cart_item_key}"
                        title="Remove from cart">×</button>
            </div>
        `;
    }

    setupBasketRemoveListeners() {
        const removeButtons = document.querySelectorAll('.basket-remove-btn');
        removeButtons.forEach(button => {
            button.addEventListener('click', async (e) => {
                e.preventDefault();
                e.stopPropagation();

                const cartItemKey = button.getAttribute('data-cart-item-key');
                void 0;

                try {
                    const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
                    const response = await runtime.sendMessage({
                        action: 'removeFromBasket',
                        cartItemKey: cartItemKey
                    });

                    if (response && response.success) {
                        void 0;
                        this.displayBasket(response.basket);
                    } else {
                        console.error('CarbonGuru: Failed to remove item from cart:', response);
                        alert('Failed to remove item from cart');
                    }
                } catch (error) {
                    console.error('CarbonGuru: Error removing item from cart:', error);
                    alert('Error removing item from cart');
                }
            });
        });
    }

    async refreshBasketDisplay() {
        await this.loadBasket();
    }

    /**
     * Fetch offset and removal costs for recent items and update button displays
     */
    async fetchRecentItemCosts(items) {
        const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
        const container = document.getElementById('recent-items-list');

        for (let index = 0; index < items.length; index++) {
            const item = items[index];
            const carbon_kg = item.product_carbon_footprint;

            if (!carbon_kg || carbon_kg <= 0) {
                continue;
            }

            try {
                const costs = await runtime.sendMessage({
                    action: 'getOffsetCosts',
                    carbon_kg: carbon_kg
                });

                if (costs && costs.success) {
                    // Find buttons for this item
                    const offsetBtn = container.querySelector(`.offset-btn[data-item-index="${index}"]`);
                    const removalBtn = container.querySelector(`.removal-btn[data-item-index="${index}"]`);

                    if (offsetBtn) {
                        const costSpan = offsetBtn.querySelector('.btn-cost');
                        if (costSpan && costs.offset_cost_usd > 0) {
                            costSpan.textContent = formatCost(costs.offset_cost_usd);
                        } else if (costSpan) {
                            costSpan.textContent = 'N/A';
                        }
                    }

                    if (removalBtn) {
                        const costSpan = removalBtn.querySelector('.btn-cost');
                        if (costSpan && costs.removal_cost_usd > 0) {
                            costSpan.textContent = formatCost(costs.removal_cost_usd);
                        } else if (costSpan) {
                            costSpan.textContent = 'N/A';
                        }
                    }
                }
            } catch (error) {
                void 0;
            }
        }
    }

    /**
     * Fetch and update costs for basket items that are missing prices
     */
    async fetchBasketItemCosts(items) {
        const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
        const container = document.getElementById('basket-items-list');
        if (!container) return;

        for (const item of items) {
            const carbon_kg = item.product_carbon_footprint * (item.quantity || 1);
            const isRemoval = item.preference_type === 'removal';

            // Check if cost is missing for this item's type
            const hasCost = isRemoval ? item.removal_cost_usd : (item.offset_cost_usd ?? item.display_cost_usd ?? item.orderable_cost_usd);

            if (!carbon_kg || carbon_kg <= 0 || hasCost) {
                continue; // Skip if no carbon or already has cost
            }

            try {
                const costs = await runtime.sendMessage({
                    action: 'getOffsetCosts',
                    carbon_kg: carbon_kg
                });

                if (costs && costs.success) {
                    // Find the basket item element by cart_item_key
                    const itemEl = container.querySelector(`.basket-item[data-cart-item-key="${item.cart_item_key}"]`);
                    if (itemEl) {
                        const footprintEl = itemEl.querySelector('.basket-item-footprint');
                        if (footprintEl) {
                            // Get the cost based on item type
                            const cost = isRemoval ? costs.removal_cost_usd : costs.offset_cost_usd;
                            if (cost > 0) {
                                // Check if cost span already exists
                                let costSpan = footprintEl.querySelector('.cost');
                                if (!costSpan) {
                                    // Create cost span if it doesn't exist
                                    costSpan = document.createElement('span');
                                    costSpan.className = `cost ${isRemoval ? 'removal-cost' : 'offset-cost'}`;
                                    footprintEl.appendChild(costSpan);
                                }
                                costSpan.textContent = formatCost(cost);
                            }
                        }
                    }
                }
            } catch (error) {
                void 0;
            }
        }
    }

    formatFootprint(footprint) {
        const value = parseFloat(footprint) || 0;
        if (value === 0) return '0 kg CO₂e';
        if (value < 0.01) return '<0.01 kg CO₂e';
        if (value < 1) return `${value.toFixed(2)} kg CO₂e`;
        if (value < 1000) return `${value.toFixed(1)} kg CO₂e`;
        return `${(value / 1000).toFixed(1)} t CO₂e`;
    }

    async offsetAllItems() {
        try {
            void 0;
            const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
            const response = await runtime.sendMessage({
                action: 'getOffsetUrl'
            });

            if (response && response.success && response.url) {
                // Open offset page in new tab
                const tabsAPI = typeof browser !== 'undefined' ? browser.tabs : chrome.tabs;
                tabsAPI.create({ url: response.url });
                void 0;
            } else {
                console.error('CarbonGuru: Failed to generate offset URL:', response);
                alert('Failed to generate offset URL');
            }
        } catch (error) {
            console.error('CarbonGuru: Error opening offset page:', error);
            alert('Error opening offset page');
        }
    }

    async clearBasket() {
        // Legacy method - redirects to WooCommerce cart
        void 0;
        const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
        await runtime.sendMessage({ action: 'openCart' });
    }

    async syncToWooCommerce() {
        // Legacy method - redirects to WooCommerce cart
        void 0;
        const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
        await runtime.sendMessage({ action: 'openCart' });
    }

    async openWooCommerceCart() {
        try {
            void 0;
            const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
            const response = await runtime.sendMessage({ action: 'getBasket' });

            if (response && response.success && response.basket && response.basket.cart_url) {
                const tabsAPI = typeof browser !== 'undefined' ? browser.tabs : chrome.tabs;
                tabsAPI.create({ url: response.basket.cart_url });
                void 0;
            } else {
                console.error('CarbonGuru: Failed to get cart URL');
                alert('Failed to open cart');
            }
        } catch (error) {
            console.error('CarbonGuru: Error opening cart:', error);
            alert('Error opening cart');
        }
    }

    async openWooCommerceCheckout() {
        try {
            void 0;
            const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
            const response = await runtime.sendMessage({ action: 'getBasket' });

            if (response && response.success && response.basket && response.basket.checkout_url) {
                const tabsAPI = typeof browser !== 'undefined' ? browser.tabs : chrome.tabs;
                tabsAPI.create({ url: response.basket.checkout_url });
                void 0;
            } else {
                console.error('CarbonGuru: Failed to get checkout URL');
                alert('Failed to open checkout');
            }
        } catch (error) {
            console.error('CarbonGuru: Error opening checkout:', error);
            alert('Error opening checkout');
        }
    }

    /**
     * Open type-specific cart (offset or removal)
     * @param {string} type - 'offset' or 'removal'
     */
    async openTypeSpecificCart(type) {
        try {
            void 0;
            const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
            const response = await runtime.sendMessage({ action: 'getBasket' });

            if (response && response.success && response.basket && response.basket.cart_url) {
                const baseUrl = response.basket.cart_url.split('?')[0]; // Remove any existing query params
                const typeUrl = `${baseUrl}?type=${type}`;
                const tabsAPI = typeof browser !== 'undefined' ? browser.tabs : chrome.tabs;
                tabsAPI.create({ url: typeUrl });
                void 0;
            } else {
                console.error('CarbonGuru: Failed to get cart URL');
                alert('Failed to open cart');
            }
        } catch (error) {
            console.error(`CarbonGuru: Error opening ${type} cart:`, error);
            alert('Error opening cart');
        }
    }

    /**
     * Open type-specific checkout (offset or removal)
     * @param {string} type - 'offset' or 'removal'
     */
    async openTypeSpecificCheckout(type) {
        try {
            void 0;
            const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
            const response = await runtime.sendMessage({ action: 'getBasket' });

            if (response && response.success && response.basket && response.basket.checkout_url) {
                const baseUrl = response.basket.checkout_url.split('?')[0]; // Remove any existing query params
                const typeUrl = `${baseUrl}?type=${type}`;
                const tabsAPI = typeof browser !== 'undefined' ? browser.tabs : chrome.tabs;
                tabsAPI.create({ url: typeUrl });
                void 0;
            } else {
                console.error('CarbonGuru: Failed to get checkout URL');
                alert('Failed to open checkout');
            }
        } catch (error) {
            console.error(`CarbonGuru: Error opening ${type} checkout:`, error);
            alert('Error opening checkout');
        }
    }

    /**
     * No-op - sync status no longer needed (WooCommerce is single source of truth)
     */
    async updateSyncStatus() {
        // Legacy method - extension now directly displays WooCommerce cart
        void 0;
    }

    /**
     * No-op - manual sync no longer needed (WooCommerce is single source of truth)
     */
    async performManualSync() {
        // Legacy method - extension now directly displays WooCommerce cart
        void 0;
        await this.loadBasket(); // Just refresh the display
    }

    /**
     * Start background polling when popup opens
     */
    async syncOnPopupOpen() {
        try {
            void 0;

            const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;

            // Start background polling
            await runtime.sendMessage({
                action: 'startPolling'
            });

            void 0;

        } catch (error) {
            console.error('CarbonGuru: Error starting cart polling:', error);
        }
    }


    // Toggle admin section
    toggleAdminSection() {
        const adminContent = document.getElementById('admin-content');
        const adminToggle = document.getElementById('admin-toggle');
        const toggleIcon = document.querySelector('.toggle-icon');

        if (adminContent && toggleIcon) {
            if (adminContent.style.display === 'none') {
                adminContent.style.display = 'block';
                toggleIcon.textContent = '▲';
                toggleIcon.style.transform = 'rotate(180deg)';
                if (adminToggle) adminToggle.setAttribute('aria-expanded', 'true');

                // Load admin info when opening
                this.loadVersionInfo();
                this.checkAuthStatus();
            } else {
                adminContent.style.display = 'none';
                toggleIcon.textContent = '▼';
                toggleIcon.style.transform = 'rotate(0deg)';
                if (adminToggle) adminToggle.setAttribute('aria-expanded', 'false');
            }
        }
    }

    // OAuth Authentication functions
    async initiateOAuth() {
        void 0;
        const authStatus = document.getElementById('auth-status');
        const authError = document.getElementById('auth-error');
        const authBtn = document.getElementById('auth-btn');

        try {
            if (authStatus) authStatus.textContent = 'Opening authorization page...';
            if (authError) authError.style.display = 'none';
            if (authBtn) authBtn.disabled = true;

            const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
            const response = await runtime.sendMessage({
                action: 'initiateOAuth'
            });

            void 0;

            if (response && response.success) {
                if (authStatus) authStatus.textContent = 'Please complete authorization in the opened tab...';
                void 0;

                // Note: Actual authentication will be completed when user authorizes
                // and the callback is processed by the background script
            } else {
                const errorMsg = response?.error || 'Failed to initiate OAuth';
                console.error('CarbonGuru: OAuth initiation failed:', errorMsg);
                if (authStatus) authStatus.textContent = '❌ Authentication failed';
                if (authError) {
                    authError.textContent = errorMsg;
                    authError.style.display = 'block';
                }
            }
        } catch (error) {
            console.error('CarbonGuru: OAuth initiation error:', error);
            if (authStatus) authStatus.textContent = '❌ Authentication failed';
            if (authError) {
                authError.textContent = error.message;
                authError.style.display = 'block';
            }
        } finally {
            if (authBtn) authBtn.disabled = false;
        }
    }

    async logout() {
        void 0;
        const authStatus = document.getElementById('auth-status');
        const authError = document.getElementById('auth-error');

        try {
            const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
            const response = await runtime.sendMessage({
                action: 'logout'
            });

            void 0;

            if (response && response.success) {
                void 0;
                // Update UI
                this.checkAuthStatus();
            } else {
                console.error('CarbonGuru: Logout failed');
                if (authError) {
                    authError.textContent = 'Logout failed';
                    authError.style.display = 'block';
                }
            }
        } catch (error) {
            console.error('CarbonGuru: Logout error:', error);
            if (authError) {
                authError.textContent = error.message;
                authError.style.display = 'block';
            }
        }
    }

    async checkAuthStatus() {
        const authStatus = document.getElementById('auth-status');
        const authBtn = document.getElementById('auth-btn');
        const logoutBtn = document.getElementById('logout-btn');
        const authUserInfo = document.getElementById('auth-user-info');
        const authUserEmail = document.getElementById('auth-user-email');
        const myAccountLink = document.getElementById('my-account-link');
        const gmailSettingsLink = document.getElementById('gmail-settings-link');

        try {
            const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
            const response = await runtime.sendMessage({
                action: 'checkAuthStatus'
            });

            if (response && response.isAuthenticated) {
                if (authStatus) authStatus.textContent = '✅ Authenticated';
                if (authBtn) {
                    authBtn.style.display = 'none';
                }
                if (logoutBtn) {
                    logoutBtn.style.display = 'inline-block';
                }

                // Show user info if available
                if (response.user && authUserInfo && authUserEmail) {
                    authUserEmail.textContent = response.user.email || response.user.display_name || 'User';
                    authUserInfo.style.display = 'block';
                }

                // Show My Account link with correct URL
                if (myAccountLink && response.backendUrl) {
                    myAccountLink.href = `${response.backendUrl}/my-account`;
                    myAccountLink.style.display = 'block';
                }

                // Show Gmail Settings link with correct URL
                if (gmailSettingsLink && response.backendUrl) {
                    gmailSettingsLink.href = `${response.backendUrl}/gmail-settings-page`;
                    gmailSettingsLink.style.display = 'block';
                }
            } else {
                if (authStatus) authStatus.textContent = '❌ Not authenticated';
                if (authBtn) {
                    authBtn.textContent = 'Sign in with WordPress';
                    authBtn.style.display = 'inline-block';
                }
                if (logoutBtn) {
                    logoutBtn.style.display = 'none';
                }
                if (authUserInfo) {
                    authUserInfo.style.display = 'none';
                }
                if (myAccountLink) {
                    myAccountLink.style.display = 'none';
                }
                if (gmailSettingsLink) {
                    gmailSettingsLink.style.display = 'none';
                }
            }
        } catch (error) {
            console.error('CarbonGuru: Auth status check error:', error);
            if (authStatus) authStatus.textContent = '❌ Check failed';
        }

        // Also check Gmail status
        this.checkGmailStatus();
    }

    async checkGmailStatus() {
        const gmailStatus = document.getElementById('gmail-status');
        const gmailAccountInfo = document.getElementById('gmail-account-info');
        const gmailAccountEmail = document.getElementById('gmail-account-email');
        const gmailConnectLink = document.getElementById('gmail-connect-link');

        try {
            const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
            const response = await runtime.sendMessage({
                action: 'checkGmailStatus'
            });

            void 0;

            if (response && response.connected) {
                if (gmailStatus) gmailStatus.textContent = '✅ Connected';
                if (gmailAccountInfo && gmailAccountEmail && response.email) {
                    gmailAccountEmail.textContent = response.email;
                    gmailAccountInfo.style.display = 'block';
                }
                if (gmailConnectLink) {
                    gmailConnectLink.style.display = 'none';
                }
            } else {
                if (gmailStatus) gmailStatus.textContent = '❌ Not connected';
                if (gmailAccountInfo) {
                    gmailAccountInfo.style.display = 'none';
                }
                // Show connect link with correct URL
                if (gmailConnectLink && response && response.backendUrl) {
                    gmailConnectLink.href = `${response.backendUrl}/gmail-settings-page`;
                    gmailConnectLink.style.display = 'block';
                }
            }
        } catch (error) {
            console.error('CarbonGuru: Gmail status check error:', error);
            if (gmailStatus) gmailStatus.textContent = '❌ Check failed';
        }
    }

    // Version and update functions
    async loadVersionDisplay() {
        const versionDisplay = document.getElementById('version-display');
        try {
            void 0;
            const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
            const response = await runtime.sendMessage({
                action: 'getVersionInfo'
            });

            void 0;

            if (response && versionDisplay) {
                if (response.currentVersion) {
                    // Include build number if present (Safari uses separate build_number field)
                    let versionStr = response.currentVersion;
                    if (response.buildNumber && !versionStr.includes(response.buildNumber)) {
                        versionStr = `${versionStr} (Build ${response.buildNumber})`;
                    }
                    const displayText = `v${versionStr} (${response.isDevelopment ? 'Dev' : 'Prod'})`;
                    versionDisplay.textContent = displayText;
                    void 0;
                } else {
                    console.error('CarbonGuru: No currentVersion in response:', response);
                    versionDisplay.textContent = 'Version unavailable';
                }
            } else {
                console.error('CarbonGuru: No response or no versionDisplay element');
                if (versionDisplay) versionDisplay.textContent = 'Version unavailable';
            }
        } catch (error) {
            console.error('CarbonGuru: Version display error:', error);
            if (versionDisplay) versionDisplay.textContent = 'Error loading version';
        }
    }

    async loadVersionInfo() {
        const versionInfo = document.getElementById('version-info');
        try {
            void 0;
            const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
            const response = await runtime.sendMessage({
                action: 'getVersionInfo'
            });

            void 0;

            if (response && versionInfo) {
                if (response.currentVersion) {
                    // Include build number if present (Safari uses separate build_number field)
                    let versionStr = response.currentVersion;
                    if (response.buildNumber && !versionStr.includes(response.buildNumber)) {
                        versionStr = `${versionStr} (Build ${response.buildNumber})`;
                    }
                    const displayText = `v${versionStr} ${response.isDevelopment ? 'Dev' : 'Prod'}`;
                    versionInfo.textContent = displayText;
                    void 0;
                } else {
                    console.error('CarbonGuru: No currentVersion in response:', response);
                    versionInfo.textContent = 'Version unavailable';
                }
            } else {
                console.error('CarbonGuru: No response or no versionInfo element');
                if (versionInfo) versionInfo.textContent = 'No response';
            }
        } catch (error) {
            console.error('CarbonGuru: Version info error:', error);
            if (versionInfo) versionInfo.textContent = 'Error loading';
        }
    }

    async checkForUpdates() {
        const updateStatus = document.getElementById('update-status');
        try {
            if (updateStatus) updateStatus.textContent = 'Checking...';

            const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
            const response = await runtime.sendMessage({
                action: 'forceUpdateCheck'
            });

            if (response && response.success) {
                if (updateStatus) {
                    updateStatus.textContent = 'Update check triggered';
                }
                // Reload version info
                this.loadVersionInfo();
            } else {
                if (updateStatus) updateStatus.textContent = 'Check failed';
            }
        } catch (error) {
            console.error('CarbonGuru: Update check error:', error);
            if (updateStatus) updateStatus.textContent = 'Check failed';
        }
    }

    // Session management functions
    async cancelAnalysis() {
        void 0;

        try {
            const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
            const response = await runtime.sendMessage({
                action: 'cancelActiveAnalysis'
            });

            if (response && response.success) {
                void 0;
                alert(`Cancelled ${response.cancelledCount || 0} active session(s)`);

                // Update session status display
                this.updateSessionStatus();
            } else {
                console.error('CarbonGuru: Failed to cancel analysis:', response);
                alert('No active sessions to cancel');
            }
        } catch (error) {
            console.error('CarbonGuru: Error cancelling analysis:', error);
            alert('Error cancelling analysis: ' + error.message);
        }
    }

    async updateSessionStatus() {
        const sessionStatus = document.getElementById('session-status');
        const cancelBtn = document.getElementById('cancel-analysis-btn');

        try {
            const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
            const response = await runtime.sendMessage({
                action: 'getActiveSessionCount'
            });

            if (response && response.count !== undefined) {
                if (response.count > 0) {
                    sessionStatus.textContent = `${response.count} active session(s)`;
                    if (cancelBtn) cancelBtn.style.display = 'inline-block';
                } else {
                    sessionStatus.textContent = 'No active sessions';
                    if (cancelBtn) cancelBtn.style.display = 'none';
                }
            }
        } catch (error) {
            console.error('CarbonGuru: Error getting session status:', error);
        }
    }

    // Modal dialog removed - user choice handled in content script floating tile

    // Copy existing mapping functionality
    async copyExistingMapping(existingItem, url) {
        void 0;
        
        try {
            const backendUrl = await this.getBackendUrl();
            const response = await fetch(`${backendUrl}/copy_existing_mapping`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    source_item_id: existingItem.id,
                    source_session_id: existingItem.session_id,
                    current_session_id: 'extension_session', // Extension session
                    url: url
                })
            });

            if (!response.ok) {
                throw new Error(`HTTP error ${response.status}`);
            }

            const data = await response.json();
            
            if (data.success) {
                // Show success message
                alert(`Successfully copied existing mapping for "${data.product_name}"`);
                
                // Refresh recent items to show the new copied item
                this.loadRecentItems();
                
                // Update current results if we have the data
                if (data.item_data) {
                    this.updateCurrentProductResults(data.item_data);
                }
            } else {
                throw new Error(data.error || 'Failed to copy mapping');
            }
        } catch (error) {
            console.error('CarbonGuru: Error copying existing mapping:', error);
            alert(`Error copying mapping: ${error.message}`);
        }
    }
}

// Initialize the popup with comprehensive error handling
void 0;

try {
    void 0;
    window.carbonGuruPopup = new CarbonGuruPopup();
    void 0;
} catch (error) {
    console.error('CarbonGuru: Failed to create popup instance:', error);
    console.error('CarbonGuru: Error stack:', error.stack);
    
    // Try again after window load as a fallback
    window.addEventListener('load', () => {
        void 0;
        try {
            window.carbonGuruPopup = new CarbonGuruPopup();
            void 0;
        } catch (retryError) {
            console.error('CarbonGuru: Failed to create popup instance on retry:', retryError);
        }
    });
} 
